package main

import (
	"encoding/json"
	"io/ioutil"
	"os"
	"time"

	"github.com/godbus/dbus"
	"pkg.deepin.io/daemon/sync/infrastructure/hardware"
)

const (
	cacheDir    = "/var/cache/deepin/deepin-sync-daemon"
	privacyFile = cacheDir + "/privacy.json"
)

type confirmLog struct {
	Time time.Time
}

type privacyConfirmLogs struct {
	Logs map[string]*confirmLog
}

func (helper *Helper) GetHardware() (*hardware.Hardware, *dbus.Error) {
	hw, err := hardware.GetHardware()
	if err != nil {
		return nil, dbus.NewError(DBusIFC+".GetHardware",
			[]interface{}{err.Error()})
	}
	return hw, nil
}

func (helper *Helper) getConfirmLogs() (*privacyConfirmLogs, error) {
	logs := &privacyConfirmLogs{
		Logs: make(map[string]*confirmLog),
	}
	f, err := os.Open(privacyFile)
	if nil != err {
		return logs, err
	}
	defer f.Close()

	data, err := ioutil.ReadAll(f)
	if nil != err {
		return logs, err
	}
	err = json.Unmarshal(data, &logs)
	if nil != err {
		return logs, err
	}
	return logs, nil
}

func (helper *Helper) HasConfirmPrivacy(id string) (bool, *dbus.Error) {
	logs, err := helper.getConfirmLogs()
	if err != nil {
		return false, dbus.NewError(DBusIFC+".HasConfirmPrivacy", []interface{}{err.Error()})
	}

	if _, ok := logs.Logs[id]; ok {
		return true, nil
	}
	return false, nil
}

func (helper *Helper) ConfirmPrivacy(id string) *dbus.Error {
	os.MkdirAll(cacheDir, 0644)

	logs, _ := helper.getConfirmLogs()
	logs.Logs[id] = &confirmLog{
		Time: time.Now(),
	}
	data, err := json.Marshal(logs)
	if err != nil {
		return dbus.NewError(DBusIFC+".ConfirmPrivacy", []interface{}{err.Error()})
	}

	f, err := os.OpenFile(privacyFile, os.O_CREATE|os.O_WRONLY, 0755)
	if err != nil {
		return dbus.NewError(DBusIFC+".ConfirmPrivacy", []interface{}{err.Error()})
	}

	_, err = f.Write(data)
	if err != nil {
		return dbus.NewError(DBusIFC+".ConfirmPrivacy", []interface{}{err.Error()})
	}
	err = f.Close()
	if err != nil {
		return dbus.NewError(DBusIFC+".ConfirmPrivacy", []interface{}{err.Error()})
	}
	return nil
}
