package deepinid

import (
	"sync"
	"time"

	"pkg.deepin.io/daemon/sync/infrastructure/hardware"
	"pkg.deepin.io/daemon/sync/infrastructure/log"
	"pkg.deepin.io/daemon/sync/infrastructure/utils"

	"github.com/fatih/structs"
	"github.com/godbus/dbus"
	"github.com/godbus/dbus/introspect"
	"github.com/godbus/dbus/prop"
	"github.com/mitchellh/mapstructure"
)

const (
	dbusService = "com.deepin.deepinid"
	dbusPath    = "/com/deepin/deepinid"
	dbusIfc     = "com.deepin.deepinid"
)

// DeepinID save core auth data of deepin user
type DeepinID struct {
	utils.DBusProxy

	machineName string
	// tokenData   *tokenData
	hardware *hardware.Hardware
	userInfo map[string]interface{}
}

// MakePropSpec export dbus prop
func (d *DeepinID) MakePropSpec() map[string]map[string]*prop.Prop {
	return map[string]map[string]*prop.Prop{
		d.GetDBusInterface(): {
			"HardwareID": {
				Value:    &d.hardware.ID,
				Writable: false,
				Emit:     prop.EmitTrue,
				Callback: func(c *prop.Change) *dbus.Error {
					log.Info(c.Name, "changed to:", c.Value)
					return nil
				},
			},
			dbusPropMachineName: {
				Value:    &d.machineName,
				Writable: false,
				Emit:     prop.EmitTrue,
				Callback: func(c *prop.Change) *dbus.Error {
					log.Info(c.Name, "changed to:", c.Value)
					return nil
				},
			},
			dbusPropUserInfo: {
				Value:    &d.userInfo,
				Writable: false,
				Emit:     prop.EmitTrue,
				Callback: func(c *prop.Change) *dbus.Error {
					//log.Info(c.Name, "changed to:", c.Value)
					return nil
				},
			},
		},
	}
}

// MakeSignals create signals
func (d *DeepinID) MakeSignals() []introspect.Signal {
	return nil
}

// GetDBusName return dbus service name
func (d *DeepinID) GetDBusName() string {
	return dbusService
}

// GetDBusPath return dbus service path
func (d *DeepinID) GetDBusPath() dbus.ObjectPath {
	return dbusPath
}

// GetDBusInterface return dbus service interface
func (d *DeepinID) GetDBusInterface() string {
	return dbusIfc
}

// NewDeepinID create a new nsq server
func NewDeepinID(hw *hardware.Hardware) (*DeepinID, error) {
	var err error
	d := &DeepinID{
		hardware: hw,
		userInfo: make(map[string]interface{}),
	}

	td, err := readToken()
	if nil != err {
		td = &tokenData{}
		log.Infof("read cache token failed: %v", err)
	}
	td.IsLoggedIn = false
	d.userInfo = structs.Map(td)
	//log.Infof("read cache token: %v", td.UserID)

	sessionBus, err := dbus.SessionBus()
	if nil != err {
		log.Error("get session bus failed:", err)
		return nil, err
	}
	return d, d.SetupDBus(sessionBus, d)
}

// VerifyToken check token online
func (d *DeepinID) VerifyToken() {
	td := d.getTokenData()
	log.Infof("check token expired: %v %v", td.isExpired(), time.Unix(td.Expiry, 0))
	if td.isExpired() {
		d.refreshToken(td.Token, td.AccessToken, td.Expiry)
	}
	d.updateToken(td.Token, td.AccessToken, td.Expiry)
}

func (d *DeepinID) getTokenData() *tokenData {
	td := &tokenData{}
	mapstructure.Decode(d.userInfo, td)
	// log.Infof("Decode %v, %v", td, d.userInfo)
	return td
}

func (d *DeepinID) isNeedFetchToken() bool {
	td := d.getTokenData()
	return (td.isExpired() || !td.IsLoggedIn)
}

var _refreshLock sync.Mutex

func (d *DeepinID) updateToken(token, accessToken string, expiry int64) error {
	_refreshLock.Lock()
	defer _refreshLock.Unlock()

	if !d.isNeedFetchToken() {
		log.Info("skip updateToken")
		return nil
	}

	//log.Infof("%q,%q", token, accessToken)
	tokenInfo, err := getTokenInfo(token, accessToken)
	if err != nil {
		log.Errorf("check token info failed: %v", err)
		if err == errTokenNotFound {
			tokenInfo = &tokenRecvInfo{}
		} else {
			return err
		}
	}

	if len(tokenInfo.Username) != 0 {
		tokenInfo.Token = token
		tokenInfo.AccessToken = accessToken
		tokenInfo.Expiry = time.Unix(expiry, 0)
	}
	return d.doUpdateToken(tokenInfo)
}

func (d *DeepinID) refreshToken(token, accessToken string, expiry int64) error {
	_refreshLock.Lock()
	defer _refreshLock.Unlock()

	if !d.isNeedFetchToken() {
		log.Info("skip refreshToken")
		return nil
	}

	//log.Infof("%q,%q", token, accessToken)
	tokenInfo, err := refreshTokenInfo(token, accessToken)
	if err != nil {
		log.Errorf("check token info failed: %v", err)
		if err == errTokenNotFound {
			tokenInfo = &tokenRecvInfo{}
		} else {
			return err
		}
	}

	return d.doUpdateToken(tokenInfo)
}

func (d *DeepinID) doUpdateToken(tokenInfo *tokenRecvInfo) error {
	// log.Info("update local token info", tokenInfo)
	tokenData := &tokenData{
		UserID:      tokenInfo.UserID,
		Username:    tokenInfo.Username,
		Region:      tokenInfo.Region,
		Nickname:    tokenInfo.Nickname,
		Token:       tokenInfo.Token,
		AccessToken: tokenInfo.AccessToken,
		Expiry:      tokenInfo.Expiry.Unix(),
		HardwareID:  d.hardware.ID,
		IsLoggedIn:  len(tokenInfo.Username) != 0,
	}

	if len(tokenInfo.Token) != 0 {
		err := d.doSetToken(tokenData, true)
		if err != nil {
			return err
		}
	} else {
		err := removeToken()
		if err != nil {
			log.Warning("Failed to remove token:", err)
		}
	}
	m := structs.Map(tokenData)
	d.SetMapProp(&d.userInfo, dbusPropUserInfo, m)

	d.SetStrProp(&d.machineName, dbusPropMachineName, tokenInfo.MachineName)

	// log.Debug("token changed:", m)
	//log.Debug("token changed", tokenData.UserID)
	return nil
}

func (d *DeepinID) doSetToken(t *tokenData, saved bool) error {
	if !saved {
		return nil
	}
	return saveToken(t)
}
