/*
 * Copyright (C) 2020 Uniontech Technology Co., Ltd.
 *
 * Author:     gl di <diguoliang@uniontech.com>
 *
 * Maintainer: gl di <diguoliang@uniontech.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "prohibitedwindowdecision.h"

#include <iostream>
#include <unordered_map>
#include <unordered_set>

#include "utils/string.h"
#include "config/screenshotconfig.h"

bool ProhibitedWindowDecision::needProhibitScreenshot(Window window)
{
    Window root = m_windowSystem.getRootWindow();

    std::string str = ScreenShotConfig::instance()->accessOffClientClassList().join(",").toStdString();
    std::vector<std::string> vecPropertyValues = split(str, ',');

    MapWindowPropertyName2StandardValues m_unAccessInterfacePropertyStandardValues;
    m_unAccessInterfacePropertyStandardValues.emplace(std::make_pair("WM_CLASS", vecPropertyValues));

    std::vector<Window> vecUnAccessInterfaceApplicationWindow;
    if (window != 0 && window != root) {
        // get toplevel window
        Window topLevelWindow = m_windowSystem.getTopLevelWindow(window);
        if (isSpecifyWindow(topLevelWindow, m_unAccessInterfacePropertyStandardValues)) {
            vecUnAccessInterfaceApplicationWindow.emplace_back(topLevelWindow);
        }
    } else if (window == root) {
        std::vector<Window> vecTopLevelWindows = m_windowSystem.getAllTopLevelWindows();
        for (std::vector<Window>::const_iterator it = vecTopLevelWindows.begin(); it != vecTopLevelWindows.end(); it++) {
            if (isSpecifyWindow(*it, m_unAccessInterfacePropertyStandardValues)) {
                vecUnAccessInterfaceApplicationWindow.emplace_back(*it);
            }
        }
    }

    for (auto it = vecUnAccessInterfaceApplicationWindow.begin(); it != vecUnAccessInterfaceApplicationWindow.end(); ++it) {
        if (isNormalState(*it)) {
            return true;
        }
    }

    return false;
}

bool ProhibitedWindowDecision::isSpecifyWindow(Window window, const std::unordered_map<std::string, std::vector<std::string>>& mapPropertyName2Values)
{
    for (auto itt = mapPropertyName2Values.begin(); itt != mapPropertyName2Values.end(); ++itt) {
        const std::string atomName = itt->first;
        const std::vector<std::string>& setAtomStandardValues = itt->second;
        if (m_windowSystem.isViewableWindow(window)) {
            std::vector<char> windowActualPropertyValue = m_windowSystem.getWindowProperty<char>(window, atomName, XA_STRING);
            std::string strActualPropertyValue(static_cast<const char*>(&(*windowActualPropertyValue.begin())), windowActualPropertyValue.size());
            std::vector<std::string> vecPropertyValue = split(strActualPropertyValue, '\0');
            for (auto atomValuesIt = setAtomStandardValues.begin(); atomValuesIt != setAtomStandardValues.end(); atomValuesIt++) {
                if (std::find(vecPropertyValue.begin(), vecPropertyValue.end(), *atomValuesIt) != vecPropertyValue.end()) {
                    return true;
                }
            }
        }
    }

    return false;
}

bool ProhibitedWindowDecision::isNormalState(Window window)
{
    std::vector<char> windowPropertyValue = m_windowSystem.getWindowProperty<char>(window, "WM_STATE", AnyPropertyType);
    if (!windowPropertyValue.empty()) {
        if (windowPropertyValue[0] == NormalState) {
            return true;
        }
    }

    return false;
}
