// SPDX-FileCopyrightText: 2021 - 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "commonlywidget.h"

#include <DLabel>
#include <DFontSizeManager>

#include <QHBoxLayout>

#define SIZE 160, 108
#define BACKGROUND 0, 0, 0, int(0.03 * 255)

const QSize IconSize = QSize(110, 65);
const QSize ArrowSize = QSize(40, 40);

CommonlyWidget::CommonlyWidget(bool isArrow, QString svgPath, QString text, QWidget *parent)
    :DFrame (parent),
     m_isArrow(isArrow)
{
    m_hover = false;
    setFixedSize(SIZE);
    QHBoxLayout *layout = new QHBoxLayout;
    layout->setSpacing(0);
    layout->setContentsMargins(10, 0, 20, 0);
    setLayout(layout);

    m_icon = QIcon(svgPath);

    DFontSizeManager::instance()->bind(this, DFontSizeManager::T8, QFont::DemiBold);
    m_text = text;
    show();
}

void CommonlyWidget::paintEvent(QPaintEvent *event)
{
    QPainter painter(this);
    painter.setRenderHint(QPainter::SmoothPixmapTransform);
    int spech = 5;

    if (m_hover) {
        painter.setPen(Qt::NoPen);
        painter.setBrush(QBrush(QColor(0, 0, 0, 0.05*255)));
        painter.drawRoundedRect(rect(), 12, 20);
    } else {
        painter.setPen(Qt::NoPen);
        painter.setBrush(QBrush(QColor(0, 0, 0, 255 * 0.03)));
        QRect roundRect(1, 1, rect().width() - 2, rect().height() - 2);
        painter.drawRoundedRect(roundRect, 12, 20);
    }

    QSize size;
    if (!m_isArrow)
        size = IconSize;
    else
        size = ArrowSize;

        m_pixmap = m_icon.pixmap(size);

    QFontMetrics fm(painter.font());
    int textWidth = fm.horizontalAdvance(m_text);
    QSize rectsize(width(), height());
    int topSpech = (height() - (fm.height() + size.height() + spech)) / 2;
    QRect pixmapRect((width() - size.width()) / 2, topSpech, size.width(), size.height());

    painter.drawPixmap(pixmapRect, m_pixmap);

    painter.setPen(QPen(QColor(0, 0, 0, 0.8 * 255)));
    if (m_isArrow) {
        spech = 18;
        painter.setPen(QPen(QColor(0, 129, 255)));
    }
    QRect textRect(QPoint((width() - textWidth) / 2, pixmapRect.bottom() + spech), QSize(textWidth, fm.height()));
    painter.drawText(textRect, Qt::AlignCenter, m_text);
}

void CommonlyWidget::enterEvent(EnterEvent *event)
{
    m_hover = true;
    update();
    DFrame::enterEvent(event);
}

void CommonlyWidget::leaveEvent(QEvent *event)
{
    m_hover = false;
    update();
    DFrame::leaveEvent(event);
}

void CommonlyWidget::mouseReleaseEvent(QMouseEvent *event)
{
    emit clicked();
    DFrame::mouseReleaseEvent(event);
}

QString CommonlyWidget::editorText(QFontMetrics fm, QString text)
{
    QString display("...");
    for (int i = 0; i < text.size(); i++) {
        if (fm.horizontalAdvance(display) > 50) {
            break;
        }
        display.insert(i, text.at(i));
    }
    return display;
}
