// Scilab ( https://www.scilab.org/ ) - This file is part of Scilab
// Copyright (C) 2008 - INRIA - Simon LIPP <simon.lipp@scilab.org>
// Copyright (C) 2010 - DIGITEO - Pierre MARECHAL
// Copyright (C) 2010-2012 - DIGITEO - Allan CORNET
// Copyright (C) 2016 - Scilab Enterprises - Pierre-Aimé AGNEL
// Copyright (C) 2016 - Samuel GOUGEON
//
// Copyright (C) 2012 - 2016 - Scilab Enterprises
//
// This file is hereby licensed under the terms of the GNU GPL v2.0,
// pursuant to article 5.3.4 of the CeCILL v.2.1.
// This file was originally licensed under the terms of the CeCILL v2.1,
// and continues to be available under such terms.
// For more information, see the COPYING file which you should have received
// along with this program.

// Generate a loader.sce and an unloader.sce script for the toolbox

function tbx_build_loader(name, path)
    // tbx_build_loader(name, path)   // deprecated (6.0)
    // tbx_build_loader(name)         // deprecated (6.0)
    // tbx_build_loader(path)         // 6.0
    // tbx_build_loader()             // 6.0  path = pwd()

    fname = "tbx_build_loader"
    rhs = argn(2)

    // CHECKING INPUT PARAMETERS
    // -------------------------
    if and(rhs <> [0 1 2]) then
        msg = _("%s: Wrong number of input arguments: %d to %d expected.\n")
        error(msprintf(msg, fname, 0, 1))
    end

    if rhs==2
        msg = "%s: %s(name, path) is obsolete. Please use %s(path) instead.\n"
        warning(msprintf(msg, fname, fname, fname))  // no translation

    elseif rhs==0
        path = pwd()
    else
        path = name
        if type(path) <> 10 then
            msg = _("%s: Argument #%d: Text(s) expected.\n")
            error(msprintf(msg, fname, rhs))
        end
        path = path(1)
        // May be
        //  * either the former tbx_build_loader(name) (until 5.5.2)
        //  * or the new        tbx_build_loader(path) (from 6.0.0)
        if grep(path,["/" "\"])==[] && ~isdir(path) then // only name was provided
            path = pwd()
        end
        if ~isdir(path) then
            msg = _("%s: The directory ''%s'' doesn''t exist or is not read accessible.\n")
            error(msprintf(msg, fname, path))
        end
    end

    // Retrieving the toolbox name
    // ---------------------------
    name = tbx_get_name_from_path(path)
    tbx_title = name

    // Heading and trailing lines of loader.sce and unloader.sce files
    // ---------------------------------------------------------------
    headertxt = [
    "// This file is released under the 3-clause BSD license. See COPYING-BSD.",
    "// Generated by builder.sce: Please, do not edit this file",
    ""
    "oldmode = mode();      mode(-1);"
    "oldlines = lines()(2); lines(0);"
    "try"
    ]
    trailertxt = [
    "catch"
    "    [errmsg, tmp, nline, func] = lasterror()"
    "    msg = ""%s: error on line #%d: """"%s""""\n"""
    "    msg = msprintf(msg, func, nline, errmsg)"
    "    lines(oldlines)"
    "    mode(oldmode);"
    "    clear oldlines oldmode tmp nline func"
    "    error(msg);"
    "end"
    "lines(oldlines);"
    "mode(oldmode);"
    "clear oldlines oldmode;"
    "" ]

    // GENERATING loader.sce
    // ---------------------
    // * If the .start file is empty, we generate a default loader including all
    //    starting actions formerly included in ~/etc/*.start
    // * Otherwise, the classic loader calling *.start is generated
    mprintf(gettext("Generating loader.sce...\n"))
    tmp = mgetl(pathconvert(path + "/etc/")+name+".start")
    if tmp==[]
        loadertxt = mgetl(SCI+"/modules/modules_manager/etc/loader_template.sce")
        loadertxt = strsubst(loadertxt, "#TBX_NAME#", name)
        loadertxt = strsubst(loadertxt, "#TBX_TITLE#", tbx_title)
    else
        start_file = name + ".start"
        loadertxt = [
        "exec(get_absolute_file_path(""loader.sce"")+""etc"+filesep()+"""+"""+start_file+""");"
        ]
    end
    loadertxt = [headertxt ; "    " + loadertxt ; trailertxt]

    // Writting it on the disk
    loaderfile = pathconvert(path + "/loader.sce", %F);
    if ~mputl(loadertxt, loaderfile) then
        msg = _("%s: The file ""%s"" cannot been created, please check if you have write access on this file.\n")
        error(msprintf(msg, fname, loaderfile))
    end

    // GENERATING unloader.sce
    // -----------------------
    // * If the .quit file does not exist or is empty, we generate a default
    //    unloader including all actions formerly included in the default
    //    ~/etc/toolbox_skeleton.quit
    // * Otherwise, the classic unloader calling *.quit is generated

    mprintf(gettext("Generating unloader.sce...\n"))

    quit_file = listfiles(pathconvert(path + "/etc/") + "*.quit")(1)
    if ~isempty(quit_file) && mgetl(pathconvert(path + "/etc/") + name + ".quit")~=[]
        quit_file = fileparts(quit_file, "fname")
        unloadertxt = [
        "    fileQuit = get_absolute_file_path(""unloader.sce"") + ""etc" + filesep() + """ + """ + quit_file + ".quit"";"
        "    if isfile(fileQuit) then"
        "        exec(fileQuit);"
        "    end"
        ];
    else
        unloadertxt = [
        "    // Unregister the library of macros. This does not unregister its macros"
        "    clear " + name + "lib"
        "    // Del help chapter";
        "    if or(getscilabmode() == [""NW"";""STD""]) then";
        "       mprintf(""\tRemove help chapter\n"");"
        "       del_help_chapter("""+name+""", %F);"
        "    end"
        "    // Remove Preferences GUI"
        "    if getscilabmode() == ""STD"" then"
        "        removeModulePreferences(get_absolute_file_path(""unloader.sce""))"
        "    end"
        "   // TODO: detect and unlink related gateways"
        ]
    end
    unloadertxt = [ headertxt ; unloadertxt ; trailertxt ]

    // Writting the unloader.sce
    unloaderfile = pathconvert(path + "/unloader.sce", %F)
    if ~mputl(unloadertxt, unloaderfile) then
        msg = _("%s: The file ""%s"" cannot been created, please check if you have write access on this file.\n")
        error(msprintf(msg, fname, unloaderfile))
    end

endfunction
