//
// Created by uos on 2022/3/17.
//

#include "AdvancedModule.h"
#include "utils/Device.h"
#include "utils/Utils.h"
#include "utils/global.h"
#include "AdvancedWidget.h"
#include "AdvancedGhostWidget.h"
#include "common/ProgressWidget.h"
#include "common/ResultWidget.h"
#include "common/SelectPartitionDialog.h"

#include <QFileDialog>
#include <QDesktopServices>

AdvancedModule::AdvancedModule(FrameProxyInterface *frame, ComDeepinDaemonUosrecoveryInterface *interface,
                               QObject *parent)
    : QObject(parent),
      ModuleInterface(frame, interface)
{

}

AdvancedModule::~AdvancedModule()
{

}

void AdvancedModule::initialize()
{
    if (m_advancedWidget == nullptr) {
        m_advancedWidget = new AdvancedWidget();
    }

    if (m_advancedGhostWidget == nullptr) {
        m_advancedGhostWidget = new AdvancedGhostWidget();
    }

    if(m_progressWidget == nullptr) {
        m_progressWidget = new ProgressWidget("maintitle", "subtitle", "");
    }

    connect(m_advancedWidget, &AdvancedWidget::showGhostWidgetSignal, this, &AdvancedModule::onShowGhostWidget);
    connect(m_advancedGhostWidget, &AdvancedGhostWidget::cancelBtnSignal, this, &AdvancedModule::onBack);
    connect(m_advancedGhostWidget, &AdvancedGhostWidget::checkSpaceSignal, this, &AdvancedModule::onCheckSpace, Qt::QueuedConnection);
    connect(m_advancedGhostWidget, &AdvancedGhostWidget::startMakeBtnSignal, this, &AdvancedModule::onStartMake, Qt::QueuedConnection);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::ReportProgress, this, &AdvancedModule::updateProgress);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::Error, this, &AdvancedModule::onError);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::ReportSpace, this, &AdvancedModule::onSpaceChanged);
}

QString AdvancedModule::name() const
{
    return "AdvancedModule";
}

QString AdvancedModule::icons() const
{
    return QString(":/resources/icons/backup_module.png");
}

QString AdvancedModule::text() const
{
    return QString(tr("Advanced"));
}

void AdvancedModule::active()
{
    m_frameProxy->popAllWidget();
    m_frameProxy->setCurrentWidget(this, m_advancedWidget);
}

void AdvancedModule::onBack()
{
    if (m_frameProxy) {
        m_frameProxy->back();
    }
}

void AdvancedModule::onBackHome()
{
    if (nullptr != m_frameProxy) {
        m_advancedGhostWidget->setDestDirText("");
        m_advancedGhostWidget->setResultInfo("");
        m_frameProxy->backHome();
    }
}

bool AdvancedModule::onCheckSpace(const QString &selectDir)
{
    if (selectDir.isEmpty()) {
        return false;
    }

    // 执行可用空间检查
    auto reply = m_recoveryInterface->CheckGhostBackupDiskSpace(selectDir);
    reply.waitForFinished();
    if (!reply.isValid()) {
        // 处理dbus error提示
        m_advancedGhostWidget->setResultInfo(tr("Space calculating failed"));
        return false;
    }

    m_frameProxy->setMenuDisabled(true);
    m_frameProxy->setWindowFuncClose(false);
    m_frameProxy->enableBackWard(false);
    m_frameProxy->enableModule(false);

    return true;
}

void AdvancedModule::onStartMake(const QString &selectDir)
{
    auto reply = m_recoveryInterface->CreateUImg(selectDir , m_backupSizeBytes);
    reply.waitForFinished();
    if (!reply.isValid()) {
        onShowResult(false, OperateType::GhostBackup, tr("Failed to create the .uimg image file"));
        return;
    }

    // 切换到进度界面
    onShowProgress(tr("Creating the .uimg image file"),
                   tr("Time remaining:") + " ",
                   tr("Your system may be stuck in the process, please wait patiently"));

}

void AdvancedModule::onShowGhostWidget()
{
    m_advancedGhostWidget->setDestDirText("");
    m_advancedGhostWidget->setResultInfo("", false);
    m_frameProxy->setCurrentWidget(this, m_advancedGhostWidget);
}

void AdvancedModule::updateProgress(const QString &progress)
{
    QJsonObject jsonObject = Utils::QStringToJson(progress);
    int operateType = jsonObject.value("operateType").toInt();
    if (operateType != OperateType::GhostBackup) {
        return;
    }

    if (m_progressWidget != nullptr) {
        auto curProgress = jsonObject.value("progress").toInt();
        auto remainSecond = jsonObject.value("remainSecond").toInt();
        m_progressWidget->setRemainTime(remainSecond);
        m_progressWidget->setValue(curProgress);
        if (curProgress >= 100) {
            onShowResult(true, operateType, "");
        }

    //    qInfo() << "updateProgress : " << jsonObject;
    }
}

void AdvancedModule::onShowProgress(const QString &mainTitle, const QString &subTitle, const QString &warning)
{
    if (m_progressWidget == nullptr) {
        m_progressWidget = new ProgressWidget(mainTitle, subTitle, warning);
    } else {
        m_progressWidget->setMainTitle(mainTitle);
        m_progressWidget->setSubTitle(subTitle);
        m_progressWidget->setWarning(warning);
    }

    m_frameProxy->setCurrentWidget(this, m_progressWidget);

    m_frameProxy->setMenuDisabled(true);
    m_frameProxy->setWindowFuncClose(false);

    m_frameProxy->enableBackWard(false);
    m_frameProxy->enableModule(false);

    m_progressWidget->start();
}

void AdvancedModule::onShowResult(bool success, int operateType, const QString &errorMsg)
{
    QString resultText = "";
    QString btnText = "";

    if (operateType == GhostBackup) {
        resultText = success ? tr("Creation successful!") : tr("Sorry, creation failed!");
        btnText = tr("OK", "button");
    } else {
        return;
    }

    m_frameProxy->setMenuDisabled(false);
    m_frameProxy->setWindowFuncClose(true);

    if (m_progressWidget != nullptr) {
        m_frameProxy->popWidget();
    }

    if (m_resultWidget == nullptr) {
        m_resultWidget = new ResultWidget(success, resultText, errorMsg, btnText);
        connect(m_resultWidget, &ResultWidget::done, this, &AdvancedModule::onBackHome);
        connect(m_resultWidget, &ResultWidget::viewBtnClicked, this, [=](){
            // 显示ghost镜像路径
            QDesktopServices::openUrl(QUrl("file://" + m_advancedGhostWidget->getDestDirText()));
        });
    } else {
        m_resultWidget->set(success, resultText, errorMsg, btnText);
    }

    m_resultWidget->showViewButton(success, tr("View Image File"), "");

    m_frameProxy->setCurrentWidget(this, m_resultWidget);
    m_frameProxy->enableBackWard(true);
    m_frameProxy->enableModule(true);
}

void AdvancedModule::onError(const QString &errMsg)
{
    QJsonObject jsonObject = Utils::QStringToJson(errMsg);
    int opType = -1;
    if (jsonObject.contains("operateType")) {
        opType = jsonObject.value("operateType").toInt();
    }

    onShowResult(false, opType, "");
}

void AdvancedModule::onSpaceChanged(const QString &space)
{
    QJsonObject jsonObject = Utils::QStringToJson(space);
    int operateType = jsonObject.value("operateType").toInt();
    if (operateType == OperateType::CheckGhostBackupSpace) {
        // 处理提示
        int errCode = jsonObject.value("errCode").toInt();
        QString destPath = jsonObject.value("destDir").toString();
        if (errCode == OK) {
            m_backupSizeBytes = jsonObject.value("backupSizeBytes").toVariant().toLongLong();
            m_advancedGhostWidget->setDestDirText(destPath);
            m_advancedGhostWidget->setResultInfo("");
        } else if (errCode == InsufficientDiskSpace) {
            m_advancedGhostWidget->setDestDirText("");
            m_advancedGhostWidget->setResultInfo(tr("Insufficient space in the selected directory. Please reselect one."));
        } else {
            m_advancedGhostWidget->setResultInfo(jsonObject.value("errMsg").toString());
        }

        m_frameProxy->setMenuDisabled(false);
        m_frameProxy->setWindowFuncClose(true);
        m_frameProxy->enableBackWard(true);
        m_frameProxy->enableModule(true);
    }
}
