//
// Created by uos on 2022/3/16.
//

#include "RestoreModule.h"
#include "utils/Utils.h"
#include "utils/Process.h"
#include <QDateTime>
#include <QJsonValue>
#include <QJsonArray>
#include <QJsonObject>
#include <DDialog>
#include <QUuid>


RestoreModule::RestoreModule(FrameProxyInterface *frame, ComDeepinDaemonUosrecoveryInterface *interface,
                             QObject *parent)
    : QObject(parent),
      ModuleInterface(frame, interface)
{

}

RestoreModule::~RestoreModule()
{

}

void RestoreModule::initialize()
{
    if (m_restoreWidget == nullptr) {
        m_restoreWidget = new RestoreWidget;
    }

    connect(m_restoreWidget, &RestoreWidget::notifySystemRestore, this, &RestoreModule::onSystemRestore);
    connect(m_restoreWidget, &RestoreWidget::notifyDataRestore, this, &RestoreModule::onDataRestore);
    connect(m_restoreWidget, &RestoreWidget::notifyInitializeRestore, this, &RestoreModule::onInitializeRestore);
    connect(m_restoreWidget, &RestoreWidget::notifyOtherRestore, this, &RestoreModule::onOtherRestore);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::ReportProgress,
            this, &RestoreModule::updateProgress);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::Error,
            this, &RestoreModule::onError);
    connect(m_recoveryInterface, &ComDeepinDaemonUosrecoveryInterface::ReportSpace, this, &RestoreModule::onSpaceChanged);
}

QString RestoreModule::name() const
{
    return "RestoreModule";
}

QString RestoreModule::icons() const
{
    return QString(":/resources/icons/backup_module.png");
}

QString RestoreModule::text() const
{
    return QString(tr("Restore"));
}

void RestoreModule::active()
{
    m_frameProxy->popAllWidget();
    m_frameProxy->setCurrentWidget(this, m_restoreWidget);
}

void RestoreModule::onSystemRestore()
{
    if (!Utils::authorization()) {
        return;
    }

    if (m_systemRestoreWidget == nullptr) {
        m_systemRestoreWidget = new SystemRestoreWidget();
        connect(m_systemRestoreWidget, &SystemRestoreWidget::cancel, this, &RestoreModule::onBack);
        connect(m_systemRestoreWidget, &SystemRestoreWidget::start, this,
                &RestoreModule::onStartSystemRestore);
    }
    QString content;
#ifdef UI_TEST
    QString filename = TEST_DATA_DIR"system_backup_info.json";
    QFile file(filename);
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    content = file.readAll();
    file.close();
#else
    auto reply = m_recoveryInterface->ListSystemBackup();
    reply.waitForFinished();
    if (!reply.isValid()) {
        qCritical() << reply.error();
        return;
    }
    content = reply.value();
#endif
    QJsonObject backupInfo = Utils::QStringToJson(content);
    auto array = backupInfo.value("backup").toArray();
    BackupInfoList backupInfoList;
    /*
     *{
      "operateID": "",
      "username": "",
      "startTime": 1648792829306,
      "remark": "",
      "operateType": 0,
      "RecoveryType": 0,
      "backupPath": ""
      },
     */
    for (int i = 0; i < array.size(); ++i) {
        BackupInfo info;
        QJsonObject object = array.at(i).toObject();
        info.unmarshal(object);
        if (info.recoveryType == RecoveryType::OSTree) {
            info.username = tr("Administrator");
            if (info.submissionType == CommitType::SystemCommit) {
                info.remark = tr("Automatical backup before system updates");
            } else if (info.submissionType == CommitType::InstallerCommit) {
                info.remark = tr("Initial backup during system installation");
            }
        }
        backupInfoList.append(info);
    }
    m_systemRestoreWidget->setData(backupInfoList);
    m_frameProxy->setCurrentWidget(this, m_systemRestoreWidget);
}

void RestoreModule::onStartSystemRestore(BackupInfo &backupInfo)
{
    DDialog dialog;
    dialog.setTitle(tr("Restore"));
    dialog.setMessage(tr("Your current system will be restored, and your data may be affected. Are you sure you want to restore your system?"));
    dialog.setIcon(QIcon::fromTheme("uos-recovery"));
    dialog.addButton(tr("Cancel", "button"));

    int result = dialog.addButton(tr("Restore", "button"), true, DDialog::ButtonWarning);
    if (dialog.exec() != result) {
        return;
    }

    m_frameProxy->setMenuDisabled(true);
    m_frameProxy->setWindowFuncClose(false);
    m_systemRestoreWidget->setTips(tr("Preparing for system restore..."));
    m_systemRestoreWidget->setTipsStyleSheet("QLabel {"
                                                "color: #000000;"
                                                "}");
    m_systemRestoreWidget->startSpinner();

    SystemRestoreRequest request;
    request.username = Utils::getUserName();
    request.backupInfo = backupInfo;
    auto requestObject = request.marshal();
    auto reply = m_recoveryInterface->SystemRestore(Utils::JsonToQString(requestObject));
    reply.waitForFinished();
    if (!reply.isValid()) {
        m_frameProxy->setMenuDisabled(false);
        m_frameProxy->setWindowFuncClose(true);
        m_systemRestoreWidget->setTips(tr("DBus error, please try again"));
        m_systemRestoreWidget->stopSpinner();
        qCritical() << reply.error();
        return;
    }

    if (backupInfo.recoveryType != RecoveryType::OSTree) { // OSTree 是异步去处理的
        m_frameProxy->setMenuDisabled(false);
        m_frameProxy->setWindowFuncClose(true);
        m_systemRestoreWidget->setTips("");
        m_systemRestoreWidget->stopSpinner();
        this->reboot();
    }
}

int RestoreModule::reboot()
{
    if (showRebootDialog(tr("Restore"),
                         tr("System restore is ready. Do you want to reboot and restore your system now?"),
                         "uos-recovery", tr("Reboot Later"), tr("Reboot and Restore"))) {
        auto rebootReply = m_recoveryInterface->Reboot();
        rebootReply.waitForFinished();
        if (!rebootReply.isValid()) {
            qCritical() << Q_FUNC_INFO <<" call Reboot failed! " << rebootReply.error();
            return -1;
        }
    }

    return 0;
}

bool RestoreModule::showRebootDialog(const QString &title, const QString &msg, const QString &iconName,
                                     const QString &leftBtnText, const QString &rightBtnText)
{
    DDialog dialog;
    dialog.setTitle(title);
    dialog.setMessage(msg);
    dialog.setIcon(QIcon::fromTheme(iconName));
    dialog.addButton(leftBtnText);

    int result = dialog.addButton(rightBtnText, true, DDialog::ButtonWarning);
    if (dialog.exec() != result) {
        return false;
    }

    return true;
}

void RestoreModule::onDataRestore()
{
    if (!Utils::checkCommonUserAuthentication()) {
        return;
    }

    if (m_userDataRestoreWidget == nullptr) {
        m_userDataRestoreWidget = new UserDataRestoreWidget();
        connect(m_userDataRestoreWidget, &UserDataRestoreWidget::cancel, this, &RestoreModule::onBack);
        connect(m_userDataRestoreWidget, &UserDataRestoreWidget::start, this,
                &RestoreModule::onStartUserDataRestore);
    }

    QString content;
    BackupInfoList backupInfoList;
    auto reply = m_recoveryInterface->ListUserDataBackup(Utils::getUserName());
    reply.waitForFinished();
    if (!reply.isValid()) {
        qCritical() << reply.error();
        return;
    }

    content = reply.value();
    QJsonObject backupInfo = Utils::QStringToJson(content);
    auto array = backupInfo.value("backup").toArray();
    int arraySize = array.size();
    for (int i = 0; i < arraySize; ++i) {
        BackupInfo info;
        QJsonObject object = array.at(i).toObject();
        info.unmarshal(object);
        backupInfoList.append(info);
    }
    m_userDataRestoreWidget->setData(backupInfoList);
    m_frameProxy->setCurrentWidget(this, m_userDataRestoreWidget);
}

void RestoreModule::onStartUserDataRestore(BackupInfo &backupInfo)
{
    m_frameProxy->setMenuDisabled(true);
    m_frameProxy->setWindowFuncClose(false);

    UserDataRestoreRequest request;
    request.username = Utils::getUserName();
    request.backupInfo = backupInfo;
    // 调整一下结构中的rootUUID
    if (request.backupInfo.rootUUID.compare(m_frameProxy->rootUUID())) {
        request.backupInfo.rootUUID = m_frameProxy->rootUUID();
    }
    auto requestObject = request.marshal();
    auto reply = m_recoveryInterface->UserDataRestore(Utils::JsonToQString(requestObject));
    reply.waitForFinished();
    if (!reply.isValid()) {
        qCritical() << reply.error();
        onShowResult(false, UserDataRestore, tr("DBus error, please try again"));
        return;
    }
    onShowProgress(tr("Restoring..."),
                   tr("Time remaining: "),
                   tr("To avoid data loss, please do not use your computer during the process."));
}

void RestoreModule::onDimFileRestore(const QString &dimFilePath)
{
    if (m_restoreWidget->isVisible()) {
        m_restoreWidget->startCalculateTip(true, tr("Preparing for system restore..."));
    } else if (m_dimFileRestoreWidget->isVisible()) {
        m_dimFileRestoreWidget->startCalculateTip(true, tr("Preparing for system restore..."));
    }

    // 检测当前选中的目录是否满足dim文件解压要求，否则在全部设备中寻找空间满足要求的位置，如果没找到，提示用户空间不够不能进行还原
    auto getPathReply = m_recoveryInterface->CheckDimFileSpace(dimFilePath);
    getPathReply.waitForFinished();
    if (!getPathReply.isValid()) {
        onShowResult(false, DimFileRestore, tr("DBus error, please try again"));
    }

    m_frameProxy->setMenuDisabled(true);
    m_frameProxy->setWindowFuncClose(false);
    m_frameProxy->enableBackWard(false);
    m_frameProxy->enableModule(false);
}

void RestoreModule::onInitializeRestore()
{
    if (!Utils::authorization()) {
        return;
    }

    DDialog dialog;
    dialog.setTitle(tr("Reset to factory settings"));
    dialog.setMessage(tr("It will restore data on your system disk, clear your username and password, but keep your personal data, please confirm before proceeding."));
    dialog.setIcon(QIcon::fromTheme("uos-recovery"));
    dialog.addButton(tr("Cancel", "button"));

    int result = dialog.addButton(tr("Confirm and Reset"), true, DDialog::ButtonWarning);
    if (dialog.exec() != result) {
        return;
    }

    const int OSMajorVerionV23 = 23;
    int osVersion = 0;
    QSettings versionFile("/etc/os-version", QSettings::IniFormat);
    versionFile.beginGroup("Version");
    osVersion = versionFile.value("MajorVersion").toInt();
    versionFile.endGroup();

    if (osVersion == 20) {
        // 到/recovery/backup目录下寻找v20的初始化备份文件
        auto hasInitBack = m_recoveryInterface->HasInitBackDimFile();
        hasInitBack.waitForFinished();
        if (!hasInitBack.isValid()) {
            onShowResult(false, DimFileRestore, tr("DBus error, please try again"));
            return;
        }

        if (hasInitBack.value()) {
            // 执行备份文件转换，还原系统
            onDimFileRestore("/recovery/backup");
            return;
        } else {
            // 显示不存在初始化备份文件提示
            m_restoreWidget->setErrorInfo(tr("Cannot find the initial backup file"));
            return;
        }
    } else if (OSMajorVerionV23 == osVersion) {
        SystemRestoreRequest factoryRestoreReq;
        factoryRestoreReq.username = Utils::getUserName();
        QJsonObject requestObject = factoryRestoreReq.marshal();
        return this->restoreToOStreeFactory(Utils::JsonToQString(requestObject));
    }
}

void RestoreModule::restoreToOStreeFactory(const QString &request)
{
    int errCode = m_recoveryInterface->OStreeFactoryRestore(request);
    if (static_cast<int> (ErrorCode::OK) != errCode) {
        QString errorMsg = tr("Cannot find the initial backup file");
        DDialog dialog;
        dialog.setTitle(tr("Reset to factory settings"));
        dialog.setMessage(errorMsg);
        dialog.setIcon(QIcon::fromTheme("uos-recovery"));
        dialog.addButton(tr("OK", "button"), true);
        dialog.exec();
    }
}

void RestoreModule::onOtherRestore()
{
    static QString osEditionType = Utils::getOSEditionType();
    if ("Community" == osEditionType) {
        DDialog dlg;
        dlg.setMessage(tr("It is unavailable for deepin alpha version"));
        dlg.setIcon(QIcon::fromTheme("dialog-warning"));
        dlg.exec();
        return;
    }

    if (!Utils::authorization()) {
        return;
    }

    if (m_dimFileRestoreWidget == nullptr) {
        m_dimFileRestoreWidget = new dimFileRestoreWidget();
        connect(m_dimFileRestoreWidget, &dimFileRestoreWidget::cancelBtnSignal, this, &RestoreModule::onBack);
        connect(m_dimFileRestoreWidget, &dimFileRestoreWidget::startDimRestoreSignal, this, &RestoreModule::onDimFileRestore);
    }

    m_dimFileRestoreWidget->setErrorTips("");
    m_frameProxy->setCurrentWidget(this, m_dimFileRestoreWidget);
}

void RestoreModule::onBack()
{
    if (m_frameProxy) {
        m_frameProxy->back();
    }
}

void RestoreModule::onBackHome()
{
    if (m_isDimFileRestoreDone) {
        m_isDimFileRestoreDone = false;
        // 显示提示重启的弹窗
        if (showRebootDialog(tr("Restore"),
                             tr("System restore is ready. Do you want to reboot and restore your system now?"),
                             "uos-recovery", tr("Cancel", "button"), tr("Reboot and Restore"))) {
            // 重启系统
            auto rebootReply = m_recoveryInterface->Reboot();
            rebootReply.waitForFinished();
            if (!rebootReply.isValid()) {
                onShowResult(false, DimFileRestore, tr("DBus error, please try again"));
                return;
            }
        } else {
            // 还原根分区下的配置文件
            auto reply = m_recoveryInterface->ClearDimFileRestoreCfg(m_dimRestoreImgFilePath);
            reply.waitForFinished();
            if (!reply.isValid()) {
                onShowResult(false, DimFileRestore, tr("DBus error, please try again"));
                return;
            }

            if (reply.value() != OK) {
                return;
            }
        }
    }

    if (nullptr != m_frameProxy) {
        m_frameProxy->backHome();
    }
}

void RestoreModule::onShowResult(bool success, int operateType, const QString &errorMsg)
{
    QString resultText = "";
    QString btnText = tr("OK", "button");

    if (operateType == UserDataRestore || operateType == SystemRestore) {
        resultText = success ? tr("Restore successful!") : tr("Sorry, restore failed!");
    } else if (operateType == DimFileRestore) {
        resultText = success ? tr("The backup file was parsed successfully") : tr("Failed to parse the backup file");
        m_isDimFileRestoreDone = success;
    } else {
        return ;
    }

    m_frameProxy->setMenuDisabled(false);
    m_frameProxy->setWindowFuncClose(true);

    if (m_progressWidget != nullptr) {
        m_frameProxy->popWidget();
    }

    if (m_resultWidget == nullptr) {
        m_resultWidget = new ResultWidget(success, resultText, errorMsg, btnText);
        connect(m_resultWidget, &ResultWidget::done, this, &RestoreModule::onBackHome);
    } else {
        m_resultWidget->set(success, resultText, errorMsg, btnText);
    }

    m_frameProxy->setCurrentWidget(this, m_resultWidget);
    m_frameProxy->enableBackWard(true);
    m_frameProxy->enableModule(true);
}

void RestoreModule::onShowProgress(const QString &mainTitle, const QString &subTitle, const QString &warning)
{
    if (m_progressWidget == nullptr) {
        m_progressWidget = new ProgressWidget(mainTitle, subTitle, warning);
    } else {
        m_progressWidget->setMainTitle(mainTitle);
        m_progressWidget->setSubTitle(subTitle);
        m_progressWidget->setWarning(warning);
    }
    m_progressWidget->setValue(0);
    m_frameProxy->setCurrentWidget(this, m_progressWidget);
    m_frameProxy->setMenuDisabled(true);
    m_frameProxy->setWindowFuncClose(false);
    m_frameProxy->enableBackWard(false);
    m_frameProxy->enableModule(false);
    m_progressWidget->start();
}

void RestoreModule::updateProgress(const QString &progress)
{
    QJsonObject jsonObject = Utils::QStringToJson(progress);
    int operateType = jsonObject.value("operateType").toInt();
    if ((operateType != OperateType::SystemRestore) &&
        (operateType != OperateType::UserDataRestore) &&
        (operateType != OperateType::DimFileRestore)) {
        return;
    }

    auto curProgress = jsonObject.value("progress").toInt();
    if (operateType == OperateType::SystemRestore) {
        if (curProgress >= 100) {
            if (jsonObject.contains("OStreeOperate")) {
                m_frameProxy->setMenuDisabled(false);
                m_frameProxy->setWindowFuncClose(true);
                if (nullptr != m_systemRestoreWidget) {
                    m_systemRestoreWidget->setTips("");
                    m_systemRestoreWidget->stopSpinner();
                }

                static const int OSTREE_ROLLBACK = 2;
                static const int OSTREE_SUCESS = 0;
                int opType = jsonObject.value("OStreeOperate").toInt();
                int state = -1;
                if (jsonObject.contains("OStreeState")) {
                    state = jsonObject.value("OStreeState").toInt();
                }

                if ((opType == OSTREE_ROLLBACK) && (state == OSTREE_SUCESS)) {
                    // OSTree 环境下需要在接收到 StateChanged成功的信号之后重启系统
                    this->reboot();
                }
                return;
            }
        }
    }

    if (m_progressWidget != nullptr) {
        auto remainSecond = jsonObject.value("remainSecond").toInt();
        m_progressWidget->setRemainTime(remainSecond);
        m_progressWidget->setValue(curProgress);
        if (curProgress >= 100) {
            onShowResult(true, operateType);
        }
    }
}

void RestoreModule::onError(const QString &errMsg)
{
    QJsonObject jsonObject = Utils::QStringToJson(errMsg);
    QString errInfo = "Unknown";
    if (jsonObject.contains("errMsg")) {
        errInfo = jsonObject.value("errMsg").toString();
    }

    int opType = -1;
    if (jsonObject.contains("operateType")) {
        opType = jsonObject.value("operateType").toInt();
    }

    onShowResult(false, opType, errInfo);
}

void RestoreModule::onSpaceChanged(const QString &space)
{
    QJsonObject jsonObject = Utils::QStringToJson(space);
    int operateType = jsonObject.value("operateType").toInt();
    if (operateType == OperateType::CheckDimFileUseSpace) {
        // 处理提示
        QString dimFilePath = jsonObject.value("dimFilePath").toString();
        m_dimRestoreImgFilePath = jsonObject.value("imgFilePath").toString();
        if (m_dimRestoreImgFilePath.isEmpty()) {
            // 显示空间不够不能还原提示
            if (m_restoreWidget->isVisible()) {
                m_restoreWidget->startCalculateTip(false, tr("Unable to perform the system restore: insufficient disk space"));
            } else if (m_dimFileRestoreWidget->isVisible()) {
                m_dimFileRestoreWidget->startCalculateTip(false, tr("Unable to perform the system restore: insufficient disk space"));
            }

            m_frameProxy->setMenuDisabled(false);
            m_frameProxy->setWindowFuncClose(true);
            m_frameProxy->enableBackWard(true);
            m_frameProxy->enableModule(true);

        } else {
            // 显示空间不够不能还原提示
            if (m_restoreWidget->isVisible()) {
                m_restoreWidget->startCalculateTip(false, "");
            } else if (m_dimFileRestoreWidget->isVisible()) {
                m_dimFileRestoreWidget->startCalculateTip(false, "");
            }

            onShowProgress(tr("Restoring..."),
                           tr("Time remaining: "),
                           tr("To avoid data loss, please do not use your computer during the process."));

            auto reply = m_recoveryInterface->DimFileToLoopDeviceFile(dimFilePath, m_dimRestoreImgFilePath);
            reply.waitForFinished();
            if (!reply.isValid()) {
                qCritical() << reply.error();
                onShowResult(false, DimFileRestore, tr("DBus error, please try again"));
                return;
            }
        }
    }
}

void RestoreModule::setSystemSyncType(int type)
{
    m_systemSyncType = type;
}

void RestoreModule::setUserDataSyncType(int type)
{
    m_userDataSyncType = type;
}
