""" Python module for isentropic flow properties

Assumptions:
1) One dimensional isentropic flow
2) Perfect gas with constant specific heats and molecular weights

"""
"""
 * Copyright (C) 2006 Varun Hiremath.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * Authors: Varun Hiremath, Venkattraman A
 * Version 0.2

"""

from scipy import optimize
from math import *

class Isentropic:
    def __init__(self, g=1.4):
        """ g is the value of gamma (ratio of specific heats), default = 1.4
        """
        self.g = g

    def get_T_by_To_from_M(self, M):
        """Returns the value of T/To at Mach number M"""
        if M < 0:
            raise Exception('Value of M negative. M value should be positive')
            return
        else:
            return 1/(1 + (self.g-1)*0.5*M**2)

    def get_P_by_Po_from_M(self, M):
        """Returns the value of p/po at Mach number M"""
        if M < 0:
            raise Exception('Value of M negative. M value should be positive')
            return
        else:
            return (1 + (self.g -1 )*0.5*M**2)**(self.g/(1-self.g))

    def get_rho_by_rhoo_from_M(self, M):
        """Returns the value of rho/rhoo at Mach number M"""
        if M < 0:
            raise Exception('Value of M negative. M value should be positive')
            return
        else:
            return (1 + (self.g -1 )*0.5*M**2)**(1/(1-self.g))

    def get_Mstar_from_M(self, M):
        """Returns the value of M* at Mach number M"""
        if M < 0:
            raise Exception('Value of M negative. M value should be positive')
            return
        else:
            return sqrt(((self.g+1)*0.5*M**2)/(1 + (self.g-1)*0.5*M**2))

    def get_A_by_Astar_from_M(self, M):
        """Returns the value of A/A* at Mach number M"""
        if M < 0:
            raise Exception('Value of M negative. M value should be positive')
            return
        else:
            return 1.0/M*(2/(self.g+1)*(1+(self.g-1)*0.5*M**2))**((self.g+1)/(self.g-1)*0.5)

    def get_F_by_Fstar_from_M(self, M):
        """Returns the value of F/F* at Mach number M"""
        if M < 0:
            raise Exception('Value of M negative. M value should be positive')
            return
        else:
            return (1 + self.g*M**2)/(M * sqrt(2*(self.g+1)*(1+(self.g-1)*0.5 * M**2)))

    def get_AP_by_AstarPo_from_M(self, M):
        """Returns the value of Ap/A*po at Mach number M"""
        if M < 0:
            raise Exception('Value of M negative. M value should be positive')
            return
        else:
            return ((2/(self.g+1))**((self.g+1)/(self.g-1)*0.5))/(M * sqrt(1 + (self.g-1)*0.5 * M**2))

    def get_M_from_T_by_To(self, T_by_To):
        """Returns the value of M at T/To equal to T_by_To"""
        if T_by_To <= 0.0 or T_by_To >= 1.0: 
            raise Exception("T/T0 must be between 0 and 1")
            return
        else:
            return sqrt(2*((1/T_by_To)-1)/(self.g-1))

    def get_P_by_Po_from_T_by_To(self, T_by_To):
        """Returns the value of p/po at T/To equal to T_by_To"""
        if T_by_To <= 0.0 or T_by_To >= 1.0: 
            raise Exception("T/T0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_T_by_To(T_by_To)
            return self.get_P_by_Po_from_M(M)
    
    def get_rho_by_rhoo_from_T_by_To(self, T_by_To):
        """Returns the value of rho/rhoo at T/To equal to T_by_To"""
        if T_by_To <= 0.0 or T_by_To >= 1.0: 
            raise Exception("T/T0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_T_by_To(T_by_To)
            return self.get_rho_by_rhoo_from_M(M)
        
    def get_A_by_Astar_from_T_by_To(self, T_by_To):
        """Returns the value of A/A* at T/To equal to T_by_To"""
        if T_by_To <= 0.0 or T_by_To >= 1.0: 
            raise Exception("T/T0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_T_by_To(T_by_To)
            return self.get_A_by_Astar_from_M(M)

    def get_F_by_Fstar_from_T_by_To(self, T_by_To):
        """Returns the value of F/F* at T/To equal to T_by_To"""
        if T_by_To <= 0.0 or T_by_To >= 1.0: 
            raise Exception("T/T0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_T_by_To(T_by_To)
            return self.get_A_by_Astar_from_M(M)

    def get_AP_by_AstarPo_from_T_by_To(self, T_by_To):
        """Returns the value of Ap/A*po at T/To equal to T_by_To"""
        if T_by_To <= 0.0 or T_by_To >= 1.0: 
            raise Exception("T/T0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_T_by_To(T_by_To)
            return self.get_AP_by_AstarP0_from_M(M)
    
    def get_M_from_P_by_Po(self, P_by_Po):
        """Returns the value of M at p/po equal to P_by_Po"""
        if P_by_Po <= 0.0 or P_by_Po >= 1.0:
            raise Exception("p/p0 must be between 0 and 1")
            return
        else:
            return sqrt(2*((1/((P_by_Po)**((self.g-1)/self.g)))-1)/(self.g-1))

    def get_T_by_To_from_P_by_Po(self, P_by_Po):
        """Returns the value of T/To at p/po equal to P_by_Po"""
        if P_by_Po <= 0.0 or P_by_Po >= 1.0:
            raise Exception("p/p0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_P_by_Po(P_by_Po)
            return self.get_T_by_To_from_M(M)

    def get_rho_by_rhoo_from_P_by_Po(self, P_by_Po):
        """Returns the value of rho/rhoo at p/po equal to P_by_Po"""
        if P_by_Po <= 0.0 or P_by_Po >= 1.0:
            raise Exception("p/p0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_P_by_Po(P_by_Po)
            return self.get_rho_by_rhoo_from_M(M)
 
    def get_A_by_Astar_from_P_by_Po(self, P_by_Po):
        """Returns the value of A/A* at p/po equal to P_by_Po"""
        if P_by_Po <= 0.0 or P_by_Po >= 1.0:
            raise Exception("p/p0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_P_by_Po(P_by_Po)
            return self.get_A_by_Astar_from_M(M)

    def get_F_by_Fstar_from_P_by_Po(self, P_by_Po):
        """Returns the value of F/F* at p/po equal to P_by_Po"""
        if P_by_Po <= 0.0 or P_by_Po >= 1.0:
            raise Exception("p/p0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_P_by_Po(P_by_Po)
            return self.get_F_by_Fstar_from_M(M)

    def get_AP_by_AstarPo_from_P_by_Po(self, P_by_Po):
        """Returns the value of Ap/A*po at p/po equal to P_by_Po"""
        if P_by_Po <= 0.0 or P_by_Po >= 1.0:
            raise Exception("p/p0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_P_by_Po(P_by_Po)
            return self.get_AP_by_AstarPo_from_M(M)
    
    def get_M_from_rho_by_rhoo(self, rho_by_rhoo):
        """Returns the value of M at rho/rhoo equal to rho_by_rhoo"""
        if rho_by_rhoo <= 0.0 or rho_by_rhoo >= 1.0:
            raise Exception("rho/rho0 must be between 0 and 1")
            return
        else:
            return sqrt(2.*((1./(rho_by_rhoo**(self.g-1)))-1)/(self.g-1))

    def get_T_by_To_from_rho_by_rhoo(self, rho_by_rhoo):
        """Returns the value of T/To at rho/rhoo equal to rho_by_rhoo"""
        if rho_by_rhoo <= 0.0 or rho_by_rhoo >= 1.0:
            raise Exception("rho/rho0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_rho_by_rhoo(rho_by_rhoo)
            return self.get_T_by_To_from_M(M)

    def get_P_by_Po_from_rho_by_rhoo(self, rho_by_rhoo):
        """Returns the value of p/po at rho/rhoo equal to rho_by_rhoo"""
        if rho_by_rhoo <= 0.0 or rho_by_rhoo >= 1.0:
            raise Exception("rho/rho0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_rho_by_rhoo(rho_by_rhoo)
            return self.get_P_by_Po_from_M(M)
 
    def get_A_by_Astar_from_rho_by_rhoo(self, rho_by_rhoo):
        """Returns the value of A/A* at rho/rhoo equal to rho_by_rhoo"""
        if rho_by_rhoo <= 0.0 or rho_by_rhoo >= 1.0:
            raise Exception("rho/rho0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_rho_by_rhoo(rho_by_rhoo)
            return self.get_A_by_Astar_from_M(M)

    def get_F_by_Fstar_from_rho_by_rhoo(self, rho_by_rhoo):
        """Returns the value of F/F* at rho/rhoo equal to rho_by_rhoo"""
        if rho_by_rhoo <= 0.0 or rho_by_rhoo >= 1.0:
            raise Exception("rho/rho0 must be between 0 and 1")
            return
        else:
            M = self.get_M_from_rho_by_rhoo(rho_by_rhoo)
            return self.get_F_by_Fstar_from_M(M)

    def get_M_from_A_by_Astar(self, A_by_Astar):
        """Returns the two values of M possible(subsonic and supersonic) at A/A* equal to A_by_Astar"""
        if A_by_Astar < 1.0:
            raise Exception("A/A* must be greater than 1")
            return
        elif A_by_Astar == 1:
            return 1.0
        else:
            return (optimize.bisect(lambda M: self.get_A_by_Astar_from_M(M)-A_by_Astar, 0.001, 1.0), optimize.bisect(lambda M: self.get_A_by_Astar_from_M(M) - A_by_Astar, 1.0, 100))

